<?php
// REDLINE Bot Analytics - User Tracking
require_once 'config.php';

// Database file for storing analytics
$ANALYTICS_FILE = 'analytics.json';

// Get analytics data
function getAnalytics() {
    global $ANALYTICS_FILE;
    if (file_exists($ANALYTICS_FILE)) {
        return json_decode(file_get_contents($ANALYTICS_FILE), true);
    }
    return [
        'total_users' => 0,
        'active_users' => 0,
        'total_interactions' => 0,
        'users' => [],
        'daily_stats' => [],
        'feature_usage' => [
            'start' => 0,
            'logs' => 0,
            'stealer' => 0,
            'payments' => 0
        ]
    ];
}

// Save analytics data
function saveAnalytics($data) {
    global $ANALYTICS_FILE;
    file_put_contents($ANALYTICS_FILE, json_encode($data, JSON_PRETTY_PRINT));
}

// Track user interaction
function trackUser($chat_id, $username, $action, $data = []) {
    $analytics = getAnalytics();
    $today = date('Y-m-d');
    
    // Initialize user if new
    if (!isset($analytics['users'][$chat_id])) {
        $analytics['users'][$chat_id] = [
            'chat_id' => $chat_id,
            'username' => $username,
            'first_seen' => date('Y-m-d H:i:s'),
            'last_seen' => date('Y-m-d H:i:s'),
            'total_interactions' => 0,
            'actions' => []
        ];
        $analytics['total_users']++;
    }
    
    // Update user data
    $analytics['users'][$chat_id]['last_seen'] = date('Y-m-d H:i:s');
    $analytics['users'][$chat_id]['total_interactions']++;
    $analytics['users'][$chat_id]['actions'][] = [
        'action' => $action,
        'timestamp' => date('Y-m-d H:i:s'),
        'data' => $data
    ];
    
    // Update daily stats
    if (!isset($analytics['daily_stats'][$today])) {
        $analytics['daily_stats'][$today] = [
            'date' => $today,
            'new_users' => 0,
            'active_users' => 0,
            'interactions' => 0
        ];
    }
    
    $analytics['daily_stats'][$today]['interactions']++;
    
    // Update feature usage
    if (isset($analytics['feature_usage'][$action])) {
        $analytics['feature_usage'][$action]++;
    }
    
    $analytics['total_interactions']++;
    
    // Count active users (last 24 hours)
    $analytics['active_users'] = 0;
    foreach ($analytics['users'] as $user) {
        if (strtotime($user['last_seen']) > strtotime('-24 hours')) {
            $analytics['active_users']++;
        }
    }
    
    saveAnalytics($analytics);
}

// Get user statistics
function getUserStats() {
    $analytics = getAnalytics();
    $stats = [
        'total_users' => $analytics['total_users'],
        'active_users' => $analytics['active_users'],
        'total_interactions' => $analytics['total_interactions'],
        'feature_usage' => $analytics['feature_usage'],
        'recent_users' => [],
        'top_users' => []
    ];
    
    // Get recent users (last 7 days)
    $recent_users = [];
    foreach ($analytics['users'] as $user) {
        if (strtotime($user['last_seen']) > strtotime('-7 days')) {
            $recent_users[] = $user;
        }
    }
    
    // Sort by last seen
    usort($recent_users, function($a, $b) {
        return strtotime($b['last_seen']) - strtotime($a['last_seen']);
    });
    
    $stats['recent_users'] = array_slice($recent_users, 0, 10);
    
    // Get top users by interactions
    $top_users = $analytics['users'];
    usort($top_users, function($a, $b) {
        return $b['total_interactions'] - $a['total_interactions'];
    });
    
    $stats['top_users'] = array_slice($top_users, 0, 10);
    
    return $stats;
}

// Get daily statistics
function getDailyStats($days = 7) {
    $analytics = getAnalytics();
    $daily_stats = [];
    
    for ($i = $days - 1; $i >= 0; $i--) {
        $date = date('Y-m-d', strtotime("-$i days"));
        if (isset($analytics['daily_stats'][$date])) {
            $daily_stats[] = $analytics['daily_stats'][$date];
        } else {
            $daily_stats[] = [
                'date' => $date,
                'new_users' => 0,
                'active_users' => 0,
                'interactions' => 0
            ];
        }
    }
    
    return $daily_stats;
}

// Display analytics dashboard
function displayAnalytics() {
    $stats = getUserStats();
    $daily_stats = getDailyStats(7);
    
    echo "<!DOCTYPE html>";
    echo "<html><head><title>REDLINE Bot Analytics</title>";
    echo "<style>";
    echo "body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }";
    echo ".container { max-width: 1200px; margin: 0 auto; }";
    echo ".card { background: white; padding: 20px; margin: 10px 0; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }";
    echo ".stat { display: inline-block; margin: 10px; padding: 15px; background: #007bff; color: white; border-radius: 5px; text-align: center; }";
    echo ".stat h3 { margin: 0; font-size: 24px; }";
    echo ".stat p { margin: 5px 0 0 0; font-size: 14px; }";
    echo "table { width: 100%; border-collapse: collapse; margin: 10px 0; }";
    echo "th, td { padding: 10px; text-align: left; border-bottom: 1px solid #ddd; }";
    echo "th { background: #f8f9fa; font-weight: bold; }";
    echo ".refresh { background: #28a745; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px; display: inline-block; margin: 10px 0; }";
    echo "</style></head><body>";
    
    echo "<div class='container'>";
    echo "<h1>🚀 REDLINE Bot Analytics</h1>";
    
    // Main statistics
    echo "<div class='card'>";
    echo "<h2>📊 Overview</h2>";
    echo "<div class='stat'>";
    echo "<h3>" . $stats['total_users'] . "</h3>";
    echo "<p>Total Users</p>";
    echo "</div>";
    echo "<div class='stat'>";
    echo "<h3>" . $stats['active_users'] . "</h3>";
    echo "<p>Active Users (24h)</p>";
    echo "</div>";
    echo "<div class='stat'>";
    echo "<h3>" . $stats['total_interactions'] . "</h3>";
    echo "<p>Total Interactions</p>";
    echo "</div>";
    echo "</div>";
    
    // Feature usage
    echo "<div class='card'>";
    echo "<h2>🎯 Feature Usage</h2>";
    echo "<table>";
    echo "<tr><th>Feature</th><th>Usage Count</th></tr>";
    foreach ($stats['feature_usage'] as $feature => $count) {
        echo "<tr><td>" . ucfirst($feature) . "</td><td>" . $count . "</td></tr>";
    }
    echo "</table>";
    echo "</div>";
    
    // Recent users
    echo "<div class='card'>";
    echo "<h2>👥 Recent Users (Last 7 Days)</h2>";
    echo "<table>";
    echo "<tr><th>Username</th><th>Chat ID</th><th>Last Seen</th><th>Interactions</th></tr>";
    foreach ($stats['recent_users'] as $user) {
        echo "<tr>";
        echo "<td>@" . ($user['username'] ?: 'Unknown') . "</td>";
        echo "<td>" . $user['chat_id'] . "</td>";
        echo "<td>" . $user['last_seen'] . "</td>";
        echo "<td>" . $user['total_interactions'] . "</td>";
        echo "</tr>";
    }
    echo "</table>";
    echo "</div>";
    
    // Daily stats
    echo "<div class='card'>";
    echo "<h2>📈 Daily Statistics (Last 7 Days)</h2>";
    echo "<table>";
    echo "<tr><th>Date</th><th>New Users</th><th>Active Users</th><th>Interactions</th></tr>";
    foreach ($daily_stats as $day) {
        echo "<tr>";
        echo "<td>" . $day['date'] . "</td>";
        echo "<td>" . $day['new_users'] . "</td>";
        echo "<td>" . $day['active_users'] . "</td>";
        echo "<td>" . $day['interactions'] . "</td>";
        echo "</tr>";
    }
    echo "</table>";
    echo "</div>";
    
    echo "<a href='?' class='refresh'>🔄 Refresh Data</a>";
    echo "</div></body></html>";
}

// Handle analytics display
if (isset($_GET['view']) && $_GET['view'] === 'analytics') {
    displayAnalytics();
    exit;
}
?>
